(* ::Subsubsection:: *)
(*BakerSchmidtCameraCP package*)

(* ::Text:: *)
(*This impements the Backer-Schmidt Camera with conic primary optical system*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/BakerSchmidtCameraCP"}]*)

BeginPackage["GeometricOptics`Packages`BakerSchmidtCameraCP`", {"GeometricOptics`"}]

Options[BakerSchmidtCameraCP] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
BakerSchmidtCameraCP::usage="BakerSchmidtCameraCP calculates parameters for a Backer-Schmidt Camera optical system.";

BakerSchmidtCameraCPInputPanel::usage = "BakerSchmidtCameraCPInputPanel gives the GUI panel to work with BakerSchmidtCameraCP";

$BakerSchmidtCameraCPInputVariables = {"f1", "\[Gamma]", "\[Beta]", "diam", "\[Theta]"};

Begin["`Private`"]

BakerSchmidtCameraCP[{f1_, gamma_, beta_, diam_, theta_}, opts___]:=BakerSchmidtCameraCP[f1, gamma, beta, diam, theta, opts];

BakerSchmidtCameraCP[f1_, gamma_, beta_, diam_, theta_, OptionsPattern[]] := 
 Module[{ftot, bf, delta1, a40, K20, rad, thick, ind, costasf, sol, outputs, inPanel, outPanel},
 
		If[ArgumentsQ["BakerSchmidtCameraCP", {$BakerSchmidtCameraCPInputVariables, {f1, gamma, beta, diam, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 1, "PackageName" -> "BakerSchmidtCameraCP", "Arguments" -> <|"f1" -> f1, "\[Gamma]" -> gamma, "\[Beta]" -> beta, "diam" -> diam, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];
			
			(* Total focal length, back focal, and distance between the corrector and the primary mirror *)
			ftot = (f1 gamma)/(-2 + 2 beta + gamma);
			bf = (f1 (0.48 - 0.2 gamma))/(-0.8` + gamma);
			delta1 = (2 beta + gamma)/(-2 + 2 beta + gamma) f1;
			(* Aspheric constant of Schmidt corrector and conic constant  *)
			a40 = ReplaceAll[-(16 beta^4 - 2 (-2 + gamma)^3 + 8 beta^3 (-8 + 3 gamma) + 2 beta^2 (48 - 36 gamma + 5 gamma^2) + beta (-64 + 72 gamma - 22 gamma^2 + gamma^3))/(32 f1^3 (-1 + enne) gamma^3),
								enne -> 1.518722];
			K20 = 1/(8 (-1 + beta)^3) (-8 + 8 beta^3 + 8 beta^2 (-3 + gamma) + 8 gamma - 4 gamma^2 + gamma^3 + 2 beta (12 - 8 gamma + gamma^2));
			(* Eliminate the chromatic aberration by varying the radius of the second surface of Schmidt corrector *)
			rad = {10^9, Rc, -2 f1, -gamma f1}; 
			thick = {4, delta1, -0.6 f1};
			ind = {{1, 1.518722, 1, -1, 1}, {1, 1.522829, 1, -1, 1}, {1, 1.514719, 1, -1, 1}};
			costasf = {0, {a40, as}, 0, K20};
			TotalAberrations[rad, thick, ind, costasf, diam/2, 0, 0, -Infinity, x, theta, {lambda1, lambda2, lambda3}];
			sol = FindRoot[GOdistancegauss[[2, 4]] - GOdistancegauss[[3, 4]] == -0.05, {Rc, -5*10^6}];
			outputs = {{"Primary radius", -2f1},
						{"Secondary radius", -gamma f1},
						{"Distance between mirrors", -beta f1},
						{"Total focal length", ftot},
						{"Back focal", bf},
						{"Distance between corrector and primary mirror", delta1},
						{"Aspheric constant of corrector", a40},
						{"Conic constant of secondary mirror", K20},
						{"Radius of second surface of corrector", Rc/.sol},
						{"Height of the image", GOimageHeight[[1, GOn]] /. sol},
						{"Spherical aberration", GOaberration[[1]] /. sol},
						{"Coma", GOcoma[[1]] /. sol},
						{"Astigmatism", GOastigmatism[[1]] /. sol},
						{"Curvature", GOcurvature[[1]] /. sol}}//N;
			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Primary focal length" , "f1", f1},
							{"Coefficient \[Gamma] to calculate the radius of secondary mirror (-\[Gamma] f1)", "gamma", gamma},
							{"Coefficient \[Beta] = 1-\[Alpha], where \[Alpha] is the obstruction factor (\[Alpha] \[GreaterEqual] 0.40)", "beta", beta},
							{"Diameter of primary mirror", "diam", diam},
							{"Field angle  in degree", "theta", theta}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];
				
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "BakerSchmidtCameraCP" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "BakerSchmidtCameraCP", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "BakerSchmidtCameraCP" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "BakerSchmidtCameraCP", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1" , f1},
											 {"\[Gamma]", gamma},
											 {"\[Beta]", beta},
											 {"diam", diam},
											 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[
					TextCell[{-2f1, -gamma f1, -beta f1, ftot, bf, delta1, a40, K20, Rc/.sol, GOimageHeight[[1, GOn]] /. sol, 
							GOaberration[[1]] /. sol, GOcoma[[1]] /. sol, GOastigmatism[[1]] /. sol, N[GOcurvature[[1]] /. sol]}, "Output"]]],
							
			(* Arguments are not correct *)
			MessageDialog["BakerSchmidtCameraCP not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"]]];

BakerSchmidtCameraCPInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "BakerSchmidtCameraCP"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, gamma, beta, diam, theta} = ReplaceAll[$BakerSchmidtCameraCPInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run "<>package<>" function", "Text"],
					Grid[{{"Primary focal length" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the value of primary focal length"]},
						  {"Coefficient \[Gamma] to calculate the radius of secondary mirror (-\[Gamma] f1)", "\[Gamma]", Tooltip[InputField[Dynamic[gamma], Alignment -> Center], "Insert the value of \[Gamma]"]},
						  {"Coefficient \[Beta] = 1-\[Alpha], where \[Alpha] is the obstruction factor (\[Alpha] \[GreaterEqual] 0.40)", "\[Beta]", Tooltip[InputField[Dynamic[beta], Alignment -> Center], "Insert the value of \[Beta]"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, gamma, beta, diam, theta}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, gamma, beta, diam, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]